###############################################################
# TPM 2.0 Sample Provisioning Script to create and write PO Index
# Copyright Intel Corporation 2015
# Last update: May 12, 2015
#   Dec 2014 added algorithm agility
#   May 2015 added platform agility and support for new NV Index Handles
##########################################################################
##########################################################################
# Usage:                                                                 #
#     Tpm2PoProv.nsh   <alg>  <fileNamePrefix>  [ {-D |-C} [-NE] ]       #
#  Where:                                                                #
#      <alg> is  a supported hash algorithm                              #
#         Such as SHA1, SHA256, SHA384, SHA512, SM3                      #
#         <alg> will be substituted for %1 in Tpm2ProvTool command lines #
#      <fileNamePrefix> is the file name prefix you used to create the   #
#         def files - typically a project name or platform name          #
#        (e.g., MyFirst; Example; Purley)                                #
#         <fileNamePrefix> will be substituted for %2 in command lines   #
#         For example %2Po_%1.iDef --> ExamplePo_Sha256.iDef             #
#                     %2Po_%1.iDef --> MyFirstPo_Sha384.iDef             #
#      -D will delete the NV Index otherwise it will be created          # 
#      -NE is an optional argument for when OwnerAuth is not EMPTY       # 
##########################################################################

echo -OFF 
# Copyrights:
# You may copy, distribute, and make derivatives of this script under the 
# following provisions:
#  You must not remove or modify the Intel copyright notice nor these requirements
#  The script must only be used to provision a TPM for use with Intel Technologies
#    For any other use you must get written permission from Intel Corporation
 
##########################################################################
# IF param 3 == -D, this script will delete the NV Index
# Otherwise, this script will check if the NV Index already exists
# - if Index does not exist the script will create and write the Index 
# - if it already exists, - checks its attributes and NV Data
#   -- if wrong attributes, - deletes the index, recreates it, & writes it
#   -- if data is incorrect, - re-writes the NV data

##########################################################################
# Here is the list of files that are needed
## //User Modified Files -----------[example for %1=SHA256, %2=MyFirst]
## %2OwnerPwSession.sDef               MyFirstOwnerPwSession.sDef
## %2Po_%1.iDef                        MyFirstPo_Sha256.iDef
## Intel Provided Files -----------------------------------------------------
## EmptyAuthPwSession.sDef
## Policy%1.sDef                      PolicySha256.sDef
## Undefinespace.pDef
##########################################################################
# This script echoes progress markers to the screen and also to
#   a log file (Tpm2PoProv.log) which also captures provisioning details 
#
#  Note that there are commands indicated by "# For testing on NEX"  
#    These never get executed when the script is run in the EFI environment
#     - they are for ease of manually testing using the NSH Exerciser
#     - Those lines may be deleted  
##########################################################################

#Check that Alg parameter is present
if ZZ%1  == ZZ   then
	echo Algorithm parameter missing
	echo   Examples: 
	echo     %0 "SHA256 Example -d"
	echo     %0 "SHA384 MyFirst -c"
	echo     %0 "SHA512 Grantley -c -NE"
	goto EOF
endif
# Check if alg and Filename parameters supported
if NOT EXIST %2Po_%1.iDef then 
	echo Algorithm or filname not found
	goto EOF
endif

# Check if 3rd parameter is correct
if p%3 == p then 
		goto NEXT
endif
if p%3 == "p-C" then 
		goto P4
endif
if p%3 == "p-D" then 
		goto P4
endif
echo "3rd parameter must be either -C or -D"
goto EOF 

:P4
# Check if 4th parameter is correct
if p%4 == p then 
		goto NEXT
endif
if p%4 == "p-NE" then 
		goto NEXT
endif
echo "4th parameter must be -NE"
goto EOF 

:NEXT
# Now copy this file to the error log
echo "TPM2 PO provisioning Script " _%1 _%2 _%3 _%4 > Tpm2PoProv.log
if EXIST %0     then
	type %0     >> Tpm2PoProv.log
endif
if EXIST %0.nsh then 
	type %0.nsh >> Tpm2PoProv.log
endif
goto START
##########################################################################
# #####For testing on NEX #####
# The following commands do not need to be sent to a TPM on a 
#   target platform because BIOS starts the TPM  
#   -- rather they are here for testing on a simulator or TTV
#   --   using the NSH Exerciser
#
#  On Simulator start-up or after inserting a TPM in the TTV
Tpm2ProvTool  PowerUp      # for testing on NEX
Tpm2ProvTool  StartUpClear # for testing on NEX
#
# Before removing a TPM from the TTV
Tpm2ProvTool ShutdownClear # for testing on NEX
Tpm2ProvTool PowerDown     # for testing on NEX
##########################################################################

:START
# Start
# Uses Session 4 for Owner AuthValue password session
# Uses Session 1 for Empty AuthValue password session
# 
# Start a PW Session to use for Index Read Auth
echo ***** Starting PW Session for PlatformAuth & Index Read Auth
echo ***** Starting PW Session for PlatformAuth & Index Read Auth >> Tpm2PoProv.log
Tpm2ProvTool StartSession EmptyAuthPwSession.sDef 1 >> Tpm2PoProv.log
	
# Start a PW Session to use for OwnerAuth and Index Write Auth
echo ***** Starting PW Session for OwnerAuth 
echo ***** Starting PW Session for OwnerAuth  >> Tpm2PoProv.log
Tpm2ProvTool StartSession EmptyAuthPwSession.sDef 4 >> Tpm2PoProv.log
if p%4 == "p-NE"  then
  Tpm2ProvTool StartSession %2OwnerPwSession.sDef 4 >> Tpm2PoProv.log
endif

# Make sure Storage hierarchy is enabled
echo **** Enabling Storage Hierarchy
echo **** Enabling Storage Hierarchy >> Tpm2PoProv.log
Tpm2ProvTool HierarchyControl 1 0x4000000C 0x40000001 1 >> Tpm2PoProv.log

echo ***************************************************************
echo *************** Provisioning PO Index *************************
echo ***************************************************************
:PO_CHECK
echo Checking if Index exists
echo **** Read NV Public to see if Index exists >> Tpm2PoProv.log
Tpm2ProvTool NvReadPublic %2PO_%1.iDef >> Tpm2PoProv.log
if not %lasterror% == 0 then	
	echo Index does not exist
        if p%3 == "p-D" then
            echo Cannot find index to delete
		    goto EOF
        endif
	goto CREATE		# Index does not exist 
endif
if p%3 == "p-D" then
    goto DELETE
endif
			
echo Comparing against definition file
echo **** Comparing against Index definition file >> Tpm2PoProv.log
Tpm2ProvTool NvVerifyDef %2PO_%1.iDef >> Tpm2PoProv.log
if %lasterror% == 0 then
  echo Verifying if NV Data is correct
  echo *** Verifying if Index NV Data is correct >> Tpm2PoProv.log
  Tpm2ProvTool NvCompareData %2PO_%1.iDef 1 >> Tpm2PoProv.log
  # If no error, PO is correct so we are done, else re-write the index
  if %lasterror% == 0 then
    goto DONE
  endif
  goto WRITE # Data is not correct
endif
# We get here because NvVerify failed -- thus attributes are incorrect, 
#   so must delete the index

# **************************************************************
:DELETE    
# **************************************************************
echo Deleting PO Index using TPM2_NV_UndefineSpace()
echo **** Deleting PO Index  >> Tpm2PoProv.log
Tpm2ProvTool NvUndefineSpace %2PO_%1.iDef 4 4 >> Tpm2PoProv.log
if not %lasterror% == 0 then
	goto ERROR		# Did not satisfy Owner/Index auth
endif
if p%3 == p-D then
    echo "******** Index Deleted ********"
    goto EOF
endif

# **************************************************************
:CREATE
# **************************************************************
echo Creating PO Index 
echo ***** Creating PO Index ***** >> Tpm2PoProv.log
Tpm2ProvTool NvDefineSpace %2PO_%1.iDef 4 >> Tpm2PoProv.log
if not %lasterror% == 0 then	
	goto ERROR		# Did not create the index
endif

# **************************************************************
:WRITE
# **************************************************************
echo Writing PO NV data 
echo ***** Writing PO Index NV Data >> Tpm2PoProv.log
Tpm2ProvTool NvWrite %2PO_%1.iDef 4 SH >> Tpm2PoProv.log
if not %lasterror% == 0 then	
#	goto ERROR		# Did not satisfy Index auth
	goto WA         # goto workaround
endif
goto DONE

################ Workaround ##################
### older tools do not support Owner Auth correctly --so use authPolicy
### assumes that authPolicy is a secret value policy OR
:WA
echo Start Policy Session
echo **** Start Policy Session >> Tpm2PoProv.log
Tpm2ProvTool StartSession Policy%1.sDef 0 >> Tpm2PoProv.log

echo Policy assertion SecretOR
echo **** Policy Assertion >> Tpm2PoProv.log
Tpm2ProvTool AssertPolicy %2SecretOR%1.pDef 0 >> Tpm2PoProv.log

echo Writing NV Data
echo **** Writing NV Data >> Tpm2PoProv.log
Tpm2ProvTool NvWrite %2Po_%1.iDef 0 >> Tpm2PoProv.log
if not %lasterror% == 0 then
	goto ERROR		# Didn't satisfy Index authPolicy
endif
goto DONE
goto PO_CHECK # for testing on NEX
############ end of workaround #########################

:DONE
echo  ***************************************************
echo  ******* Provisioning Completed Successfully *******
echo  ***************************************************

goto EOF

:ERROR
echo  ***************************************************
echo  ********!!!!!!! Provisioning FAILED !!!!!!!********
echo  ***************************************************
Tpm2ProvTool FlushSession 0 >> Tpm2PoProv.log

:EOF


